/*
 * routines for loading the fabric description from a file
 */
#include <stdio.h>

#include "libfma.h"
#include "lf_db.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#define _LF_DEFINE_TABLES_
#include "lf_fabric_db.h"
#include "lf_product_def.h"
#include "lf_switch.h"
#include "lf_fma_flags.h"

/*
 * Remove a link from the database
 */
int
lf_remove_link_from_db(
  struct lf_fabric_db *fdp,
  char *name,
  int slot,
  int port,
  int subport)
{
  static db_datum_t *spec = NULL;
  static int nc;
  db_datum_t *dp;
  int rc;
  int c;

  /* If table is empty, nothing to remove, so done. */
  if (fdp->link_tbl == NULL) {
    return 0;
  }

  /* allocate template first time through */
  if (spec == NULL) {
    spec = db_alloc_row_template(fdp->link_tbl, &nc);
    if (spec == NULL) LF_ERROR(("Cannot allocate link row template"));
  }

  /* default all fields to match all */
  for (c=0; c<nc; ++c) {
    spec[c].dd_type = DBD_MATCHALL;
  }

  /* since this link could be identified from either end, we
   * need to specify this both as link1 and as link2
   */

  if (name != NULL) {
    dp = &spec[fdp->link_colids[LF_TABLE_LINK_NAME_1]];
    dp->dd_type = DBD_STR;
    dp->dbd_str = name;
  }

  if (slot != -1) {
    dp = &spec[fdp->link_colids[LF_TABLE_LINK_SLOT_1]];
    dp->dd_type = DBD_INT;
    dp->dbd_int = slot;
  }

  if (port != -1) {
    dp = &spec[fdp->link_colids[LF_TABLE_LINK_PORT_1]];
    dp->dd_type = DBD_INT;
    dp->dbd_int = port;
  }

  if (subport != -1) {
    dp = &spec[fdp->link_colids[LF_TABLE_LINK_SUBPORT_1]];
    dp->dd_type = DBD_INT;
    dp->dbd_int = subport;
  }

  /* delete if specified as "end 1" */
  rc = db_delete_row(fdp->link_tbl, spec);
  if  (rc != 0) LF_ERROR(("Error deleting link from DB"));

  /* If matchall query, return now */
  if (name == NULL && slot == -1 && port == -1 && subport == -1) {
    return 0;
  }

  /* now, re-do in case this is "end 2" */
  spec[fdp->link_colids[LF_TABLE_LINK_NAME_1]].dd_type = DBD_MATCHALL;
  spec[fdp->link_colids[LF_TABLE_LINK_SLOT_1]].dd_type = DBD_MATCHALL;
  spec[fdp->link_colids[LF_TABLE_LINK_PORT_1]].dd_type = DBD_MATCHALL;
  spec[fdp->link_colids[LF_TABLE_LINK_SUBPORT_1]].dd_type = DBD_MATCHALL;

  dp = &spec[fdp->link_colids[LF_TABLE_LINK_NAME_2]];
  dp->dd_type = DBD_STR;
  dp->dbd_str = name;

  if (slot != -1) {
    dp = &spec[fdp->link_colids[LF_TABLE_LINK_SLOT_2]];
    dp->dd_type = DBD_INT;
    dp->dbd_int = slot;
  }

  if (port != -1) {
    dp = &spec[fdp->link_colids[LF_TABLE_LINK_PORT_2]];
    dp->dd_type = DBD_INT;
    dp->dbd_int = port;
  }

  if (subport != -1) {
    dp = &spec[fdp->link_colids[LF_TABLE_LINK_SUBPORT_2]];
    dp->dd_type = DBD_INT;
    dp->dbd_int = subport;
  }

  /* delete if specified as "end 2" */
  rc = db_delete_row(fdp->link_tbl, spec);
  if  (rc != 0) LF_ERROR(("Error deleting link from DB"));

  return 0;

 except:
  return -1;
}

/*
 * Remove a NIC given hostname and host nic_id
 */
int
lf_remove_nic_from_db(
  struct lf_fabric_db *fdp,
  char *hostname,
  int nic_id)
{
  static db_datum_t *spec = NULL;
  static int nc;
  db_datum_t *dp;
  int rc;
  int c;

  /* If table is empty, nothing to remove, so done. */
  if (fdp->nic_tbl == NULL) {
    return 0;
  }

  /* allocate template first time through */
  if (spec == NULL) {
    spec = db_alloc_row_template(fdp->nic_tbl, &nc);
    if (spec == NULL) LF_ERROR(("Cannot allocate NIC row template"));

    /* default all fields to match all */
    for (c=0; c<nc; ++c) {
      spec[c].dd_type = DBD_MATCHALL;
    }
  }

  /* specify NIC hostname */
  dp = &spec[fdp->nic_colids[LF_TABLE_NIC_HOSTNAME]];
  if (hostname != NULL) {
    dp->dd_type = DBD_STR;
    dp->dbd_str = hostname;
  } else {
    dp->dd_type = DBD_MATCHALL;
  }

  dp = &spec[fdp->nic_colids[LF_TABLE_NIC_HOST_NIC_ID]];
  if (nic_id != -1) {
    dp->dd_type = DBD_INT;
    dp->dbd_int = nic_id;
  } else {
    dp->dd_type = DBD_MATCHALL;
  }

  /* remove the NIC */
  rc = db_delete_row(fdp->nic_tbl, spec);
  if (rc != 0) LF_ERROR(("Error deleting NIC from DB"));

  return 0;

 except:
  return -1;
}

/*
 * Remove a host given hostname
 */
int
lf_remove_host_from_db(
  struct lf_fabric_db *fdp,
  char *hostname)
{
  static db_datum_t *spec = NULL;
  static int nc;
  db_datum_t *dp;
  int rc;
  int c;

  /* If table is empty, nothing to remove, so done. */
  if (fdp->host_tbl == NULL) {
    return 0;
  }

  /* allocate template first time through */
  if (spec == NULL) {
    spec = db_alloc_row_template(fdp->host_tbl, &nc);
    if (spec == NULL) LF_ERROR(("Cannot allocate host row template"));

    /* default all fields to match all */
    for (c=0; c<nc; ++c) {
      spec[c].dd_type = DBD_MATCHALL;
    }
  }

  /* specify hostname */
  dp = &spec[fdp->nic_colids[LF_TABLE_NIC_HOSTNAME]];
  if (hostname != NULL) {
    dp->dd_type = DBD_STR;
    dp->dbd_str = hostname;
  } else {
    dp->dd_type = DBD_MATCHALL;
  }

  /* remove the host */
  rc = db_delete_row(fdp->host_tbl, spec);
  if (rc != 0) LF_ERROR(("Error deleting NIC from DB"));

  return 0;

 except:
  return -1;
}

/*
 * Add a host to the database
 */
int
lf_add_host_to_db(
  struct lf_fabric_db *fdp,
  struct lf_host *hp)
{
  static db_datum_t *row = NULL;
  int rc;

  /* If no host table, create it */
  if (fdp->host_tbl == NULL) {
    rc = lf_create_host_table(fdp);
    if (rc != 0) LF_ERROR(("Creating host table"));
  }

  /* allocate template first time through */
  if (row == NULL) {
    row = db_alloc_row_template(fdp->host_tbl, NULL);
    if (row == NULL) LF_ERROR(("Cannot allocate host row template"));
  }

  /* put hostname in row */
  row[fdp->host_colids[LF_TABLE_HOST_HOSTNAME]].dbd_str = hp->hostname;
  row[fdp->host_colids[LF_TABLE_HOST_FW_TYPE]].dbd_int = hp->fw_type;
  row[fdp->host_colids[LF_TABLE_HOST_FLAGS]].dbd_int =
    (hp->fma_flags & FMA_FLAGS_NO_DB_MASK);
  row[fdp->host_colids[LF_TABLE_HOST_STATUS]].dbd_int = hp->db_status;

  /* add this row to the database */
  rc = db_add_row(fdp->host_tbl, row);
  if (rc != 0) LF_ERROR(("Error adding host %s to database", hp->hostname));

  return 0;

 except:
  return -1;
}

/*
 * Add a NIC to the database
 */
int
lf_add_nic_to_db(
  struct lf_fabric_db *fdp,
  struct lf_nic *nicp)
{
  static db_datum_t *row = NULL;
  int rc;

  /* If no nic table, create it */
  if (fdp->nic_tbl == NULL) {
    rc = lf_create_nic_table(fdp);
    if (rc != 0) LF_ERROR(("Creating nic table"));
  }

  /* allocate template first time through */
  if (row == NULL) {
    row = db_alloc_row_template(fdp->nic_tbl, NULL);
    if (row == NULL) LF_ERROR(("Cannot allocate NIC row template"));
  }

  /* put values in row */
  row[fdp->nic_colids[LF_TABLE_NIC_HOSTNAME]].dbd_str = nicp->host->hostname;
  row[fdp->nic_colids[LF_TABLE_NIC_HOST_NIC_ID]].dbd_int = nicp->host_nic_id;
  row[fdp->nic_colids[LF_TABLE_NIC_PRODUCT_ID]].dbd_str = nicp->product_id;
  row[fdp->nic_colids[LF_TABLE_NIC_SERIAL_NO]].dbd_str = nicp->serial_no;
  LF_MAC_COPY(row[fdp->nic_colids[LF_TABLE_NIC_MAC_ADDR]].dbd_mac,
              nicp->mac_addr);
  row[fdp->nic_colids[LF_TABLE_NIC_PARTITION]].dbd_int = nicp->partition;
  row[fdp->nic_colids[LF_TABLE_NIC_STATUS]].dbd_int = nicp->db_status;

  /* add this row to the database */
  rc = db_add_row(fdp->nic_tbl, row);
  if (rc != 0) LF_ERROR(("Error adding NIC %s:%d to database",
                        nicp->host->hostname, nicp->host_nic_id));

  return 0;

 except:
  return -1;
}

/*
 * Add a link to the database
 */
int
lf_add_phys_link_to_db(
  struct lf_fabric_db *fdp,
  union lf_node *np1,
  int port1,
  enum lf_db_status status)
{
  static db_datum_t *row = NULL;
  union lf_node *np2;
  int port2;
  int rc;

  /* If no link table, create it */
  if (fdp->link_tbl == NULL) {
    rc = lf_create_link_table(fdp);
    if (rc != 0) LF_ERROR(("Creating link table"));
  }

  /* allocate template first time through */
  if (row == NULL) {
    row = db_alloc_row_template(fdp->link_tbl, NULL);
    if (row == NULL) LF_ERROR(("Cannot allocate link row template"));
  }

  row[fdp->link_colids[LF_TABLE_LINK_STATUS]].dbd_int = status;

  switch (np1->ln_type) {
  case LF_NODE_NIC_XCVR:
    {
      struct lf_xcvr *xcp;
      struct lf_nic *nicp;
      int nicport;

      /* Get pointer to this xcvr and its parent NIC */
      xcp = LF_XCVR(np1);
      nicp = LF_NIC(xcp->ports[port1 + xcp->num_conns]);
      nicport = xcp->rports[port1 + xcp->num_conns];

      /* find link node - if none, we are done! */
      np2 = xcp->ports[port1];
      if (np2 == NULL) return 0;
      port2 = xcp->rports[port1];

      /* Fill in row with information about this node */
      row[fdp->link_colids[LF_TABLE_LINK_NAME_1]].dbd_str =
	  nicp->host->hostname;
      row[fdp->link_colids[LF_TABLE_LINK_SLOT_1]].dbd_int = nicp->host_nic_id;
      row[fdp->link_colids[LF_TABLE_LINK_PORT_1]].dbd_int = nicport;
      row[fdp->link_colids[LF_TABLE_LINK_SUBPORT_1]].dbd_int = port1;
    }
    break;

  case LF_NODE_LC_XCVR:
    {
      struct lf_xcvr *xcp;

      xcp = LF_XCVR(np1);

      /* find link node - if none, we are done! */
      np2 = xcp->ports[port1];
      if (np2 == NULL) return 0;
      port2 = xcp->rports[port1];

      /* Fill in row with information about this node */
      row[fdp->link_colids[LF_TABLE_LINK_NAME_1]].dbd_str =
	  xcp->p.linecard->enclosure->name;
      row[fdp->link_colids[LF_TABLE_LINK_SLOT_1]].dbd_int =
	  lf_slot_display_no(xcp->p.linecard);
      row[fdp->link_colids[LF_TABLE_LINK_PORT_1]].dbd_int =
	  xcp->p.linecard->xcvr_labels[xcp->port];
      row[fdp->link_colids[LF_TABLE_LINK_SUBPORT_1]].dbd_int = port1;
    }
    break;

  default:
    LF_ERROR(("Link type not supported!"));
  }

  switch (np2->ln_type) {
  case LF_NODE_NIC_XCVR:
    {
      struct lf_xcvr *xcp;

      xcp = LF_XCVR(np2);

      /* Fill in row with information about this node */
      row[fdp->link_colids[LF_TABLE_LINK_NAME_2]].dbd_str =
	  xcp->p.nic->host->hostname;
      row[fdp->link_colids[LF_TABLE_LINK_SLOT_2]].dbd_int = 
	  xcp->p.nic->host_nic_id;
      row[fdp->link_colids[LF_TABLE_LINK_PORT_2]].dbd_int =
	xcp->rports[xcp->num_conns + port2];
      row[fdp->link_colids[LF_TABLE_LINK_SUBPORT_2]].dbd_int = 0;
    }
    break;

  case LF_NODE_LC_XCVR:
    {
      struct lf_xcvr *xcp;

      xcp = LF_XCVR(np2);

      /* Fill in row with information about this node */
      row[fdp->link_colids[LF_TABLE_LINK_NAME_2]].dbd_str =
	  xcp->p.linecard->enclosure->name;
      row[fdp->link_colids[LF_TABLE_LINK_SLOT_2]].dbd_int =
	  lf_slot_display_no(xcp->p.linecard);
      row[fdp->link_colids[LF_TABLE_LINK_PORT_2]].dbd_int =
	  xcp->p.linecard->xcvr_labels[xcp->port];
      row[fdp->link_colids[LF_TABLE_LINK_SUBPORT_2]].dbd_int = port2;
    }
    break;

  default:
    LF_ERROR(("Link type not supported!"));
  }

  /* Add the row to the database */
  rc = db_add_row(fdp->link_tbl, row);
  if (rc != 0) LF_ERROR(("Error adding link to database"));

  return 0;

 except:
  return -1;
}

/*
 * Add a link to the database given a topo connection
 */
int
lf_add_topo_link_to_db(
  struct lf_fabric_db *fdp,
  union lf_node *tnp,
  int tport,
  enum lf_db_status status)
{
  union lf_node *pnp;
  int pport;
  int rc;

  /* Get the physical connection */
  pnp = lf_phys_for_topo(tnp, tport, &pport);
  if (pnp == NULL) {
    LF_ERROR(("Error getting physical node"));
  }

  /* If phys node is topo node, then this is an implicit link (XXX- hacky?) */
  if (pnp == tnp) {
    return 0;
  }

  /* Add the link using physical connection info */
  rc = lf_add_phys_link_to_db(fdp, pnp, pport, status);
  if (rc != 0) {
    LF_ERROR(("Unable to add phys link to DB"));
  }
  return 0;

 except:
  return -1;
}

/*
 * Remove an enclosure from the DB
 */
int
lf_remove_enclosure_from_db(
  struct lf_fabric_db *fdp,
  char *name)
{
  static db_datum_t *spec = NULL;
  static int nc;
  db_datum_t *dp;
  int rc;
  int c;

  /* If table is empty, nothing to remove, so done. */
  if (fdp->enclosure_tbl == NULL) {
    return 0;
  }

  /* allocate template first time through */
  if (spec == NULL) {
    spec = db_alloc_row_template(fdp->enclosure_tbl, &nc);
    if (spec == NULL) LF_ERROR(("Cannot allocate enclosure row template"));

    /* default all fields to match all */
    for (c=0; c<nc; ++c) {
      spec[c].dd_type = DBD_MATCHALL;
    }
  }

  /* specify enclosure name */
  dp = &spec[fdp->enclosure_colids[LF_TABLE_ENCLOSURE_NAME]];
  if (name != NULL) {
    dp->dd_type = DBD_STR;
    dp->dbd_str = name;
  } else {
    dp->dd_type = DBD_MATCHALL;
  }

  /* remove the enclosure */
  rc = db_delete_row(fdp->enclosure_tbl, spec);
  if (rc != 0) LF_ERROR(("Error deleting enclosure from DB"));

  return 0;

 except:
  return -1;
}

/*
 * Add an enclosure to the database
 */
int
lf_add_enclosure_to_db(
  struct lf_fabric_db *fdp,
  struct lf_enclosure *ep)
{
  static db_datum_t *row = NULL;
  int rc;

  /* If no enclosure table, create it */
  if (fdp->enclosure_tbl == NULL) {
    rc = lf_create_enclosure_table(fdp);
    if (rc != 0) LF_ERROR(("Creating enclosure table"));
  }

  /* allocate template first time through */
  if (row == NULL) {
    row = db_alloc_row_template(fdp->enclosure_tbl, NULL);
    if (row == NULL) LF_ERROR(("Cannot allocate enclosure row template"));
  }

  /* fill in enclosure row */
  row[fdp->enclosure_colids[LF_TABLE_ENCLOSURE_NAME]].dbd_str =
    ep->name;
  row[fdp->enclosure_colids[LF_TABLE_ENCLOSURE_PRODUCT_ID]].dbd_str =
    ep->product_id;
  row[fdp->enclosure_colids[LF_TABLE_ENCLOSURE_STATUS]].dbd_int = ep->db_status;

  /* add this row to the database */
  rc = db_add_row(fdp->enclosure_tbl, row);
  if (rc != 0) {
    LF_ERROR(("Error adding enclosure %s to database", ep->name));
  }

  return 0;

 except:
  return -1;
}

/*
 * Remove a linecard given enclosure name and slot
 * slot == -1 means all slots.
 */
int
lf_remove_linecard_from_db(
  struct lf_fabric_db *fdp,
  char *name,
  int slot)
{
  static db_datum_t *spec = NULL;
  static int nc;
  db_datum_t *dp;
  int rc;
  int c;

  /* If linecard table is empty, nothing to remove, so done. */
  if (fdp->linecard_tbl == NULL) {
    return 0;
  }

  /* allocate template first time through */
  if (spec == NULL) {
    spec = db_alloc_row_template(fdp->linecard_tbl, &nc);
    if (spec == NULL) LF_ERROR(("Cannot allocate linecard row template"));

    /* default all fields to match all */
    for (c=0; c<nc; ++c) {
      spec[c].dd_type = DBD_MATCHALL;
    }
  }

  /* specify enclosure name */
  dp = &spec[fdp->linecard_colids[LF_TABLE_LINECARD_ENCLOSURE_NAME]];
  if (name != NULL) {
    dp->dd_type = DBD_STR;
    dp->dbd_str = name;
  } else {
    dp->dd_type = DBD_MATCHALL;
  }

  dp = &spec[fdp->linecard_colids[LF_TABLE_LINECARD_ENCLOSURE_SLOT]];
  if (slot != -1) {
    dp->dd_type = DBD_INT;
    dp->dbd_int = slot;
  } else {
    dp->dd_type = DBD_MATCHALL;
  }

  /* remove the linecard */
  rc = db_delete_row(fdp->linecard_tbl, spec);
  if (rc != 0) LF_ERROR(("Error deleting linecard from DB"));

  return 0;

 except:
  return -1;
}

/*
 * Add a linecard to the database
 */
int
lf_add_linecard_to_db(
  struct lf_fabric_db *fdp,
  struct lf_linecard *lp)
{
  static db_datum_t *row = NULL;
  int rc;

  /* If no linecard table, create it */
  if (fdp->linecard_tbl == NULL) {
    rc = lf_create_linecard_table(fdp);
    if (rc != 0) LF_ERROR(("Creating linecard table"));
  }

  /* allocate template first time through */
  if (row == NULL) {
    row = db_alloc_row_template(fdp->linecard_tbl, NULL);
    if (row == NULL) LF_ERROR(("Cannot allocate linecard row template"));
  }

  /* fill in linecard row */
  row[fdp->linecard_colids[LF_TABLE_LINECARD_ENCLOSURE_NAME]].dbd_str =
    lp->enclosure->name;
  row[fdp->linecard_colids[LF_TABLE_LINECARD_ENCLOSURE_SLOT]].dbd_int =
    lf_slot_display_no(lp);
  row[fdp->linecard_colids[LF_TABLE_LINECARD_SERIAL_NO]].dbd_str =
    lp->serial_no;
  row[fdp->linecard_colids[LF_TABLE_LINECARD_PRODUCT_ID]].dbd_str =
    lp->product_id;
  row[fdp->linecard_colids[LF_TABLE_LINECARD_STATUS]].dbd_int = lp->db_status;

  /* add this row to the database */
  rc = db_add_row(fdp->linecard_tbl, row);
  if (rc != 0) {
    LF_ERROR(("Error adding linecard %s:%d to database", lp->enclosure->name,
	       lf_slot_display_no(lp)));
  }

  return 0;

 except:
  return -1;
}

/*
 * create the NIC table in the database
 */
int
lf_create_nic_table(
  struct lf_fabric_db *fdp)
{
  int rc;
  int col;
  char *col_names[LF_TABLE_NIC_NUMCOLS];
  db_datum_type_t col_types[LF_TABLE_NIC_NUMCOLS];

  /* fill in names and types */
  for (col=0; col<LF_TABLE_NIC_NUMCOLS; ++col) {
    col_names[col] = lf_table_nic_cols[col].col_name;
    col_types[col] = lf_table_nic_cols[col].col_type;

    /* column indices are identity since we created it that way */
    fdp->nic_colids[col] = col;
  }

  /* create the database */
  rc = db_create_table(fdp->database, LF_TABLE_NICS,
                             LF_TABLE_NIC_NUMCOLS, col_names, col_types);
  if (rc != 0) LF_ERROR(("Error creating nic table"));

  fdp->nic_tbl = db_open_table(fdp->database, LF_TABLE_NICS);
  if (fdp->nic_tbl == NULL) LF_ERROR(("Error opening nic table"));

  return 0;

 except:
  return -1;
}

/*
 * create the host table in the database
 */
int
lf_create_host_table(
  struct lf_fabric_db *fdp)
{
  int rc;
  int col;
  char *col_names[LF_TABLE_HOST_NUMCOLS];
  db_datum_type_t col_types[LF_TABLE_HOST_NUMCOLS];

  /* fill in names and types */
  for (col=0; col<LF_TABLE_HOST_NUMCOLS; ++col) {
    col_names[col] = lf_table_host_cols[col].col_name;
    col_types[col] = lf_table_host_cols[col].col_type;

    /* column indices are identity since we created it that way */
    fdp->host_colids[col] = col;
  }

  /* create the database */
  rc = db_create_table(fdp->database, LF_TABLE_HOSTS,
                             LF_TABLE_HOST_NUMCOLS, col_names, col_types);
  if (rc != 0) LF_ERROR(("Error creating host table"));

  fdp->host_tbl = db_open_table(fdp->database, LF_TABLE_HOSTS);
  if (fdp->host_tbl == NULL) LF_ERROR(("Error opening host table"));

  return 0;

 except:
  return -1;
}

/*
 * create the link table in the database
 */
int
lf_create_link_table(
  struct lf_fabric_db *fdp)
{
  int rc;
  int col;
  char *col_names[LF_TABLE_LINK_NUMCOLS];
  db_datum_type_t col_types[LF_TABLE_LINK_NUMCOLS];

  /* fill in names and types */
  for (col=0; col<LF_TABLE_LINK_NUMCOLS; ++col) {
    col_names[col] = lf_table_link_cols[col].col_name;
    col_types[col] = lf_table_link_cols[col].col_type;

    /* column indices are identity since we created it that way */
    fdp->link_colids[col] = col;
  }

  /* create the database */
  rc = db_create_table(fdp->database, LF_TABLE_LINKS,
                             LF_TABLE_LINK_NUMCOLS, col_names, col_types);
  if (rc != 0) LF_ERROR(("Error creating link table"));

  fdp->link_tbl = db_open_table(fdp->database, LF_TABLE_LINKS);
  if (fdp->link_tbl == NULL) LF_ERROR(("Error opening link table"));

  return 0;

 except:
  return -1;
}

/*
 * create the enclosure table in the database
 */
int
lf_create_enclosure_table(
  struct lf_fabric_db *fdp)
{
  int rc;
  int col;
  char *col_names[LF_TABLE_ENCLOSURE_NUMCOLS];
  db_datum_type_t col_types[LF_TABLE_ENCLOSURE_NUMCOLS];

  /* fill in names and types */
  for (col=0; col<LF_TABLE_ENCLOSURE_NUMCOLS; ++col) {
    col_names[col] = lf_table_enclosure_cols[col].col_name;
    col_types[col] = lf_table_enclosure_cols[col].col_type;

    /* column indices are identity since we created it that way */
    fdp->enclosure_colids[col] = col;
  }

  /* create the database */
  rc = db_create_table(fdp->database, LF_TABLE_ENCLOSURES,
                             LF_TABLE_ENCLOSURE_NUMCOLS, col_names, col_types);
  if (rc != 0) LF_ERROR(("Error creating enclosure table"));

  fdp->enclosure_tbl = db_open_table(fdp->database, LF_TABLE_ENCLOSURES);
  if (fdp->enclosure_tbl == NULL) LF_ERROR(("Error opening enclosure table"));

  return 0;

 except:
  return -1;
}

/*
 * create the linecard table in the database
 */
int
lf_create_linecard_table(
  struct lf_fabric_db *fdp)
{
  int rc;
  int col;
  char *col_names[LF_TABLE_LINECARD_NUMCOLS];
  db_datum_type_t col_types[LF_TABLE_LINECARD_NUMCOLS];

  /* fill in names and types */
  for (col=0; col<LF_TABLE_LINECARD_NUMCOLS; ++col) {
    col_names[col] = lf_table_linecard_cols[col].col_name;
    col_types[col] = lf_table_linecard_cols[col].col_type;

    /* column indices are identity since we created it that way */
    fdp->linecard_colids[col] = col;
  }

  /* create the database */
  rc = db_create_table(fdp->database, LF_TABLE_LINECARDS,
                             LF_TABLE_LINECARD_NUMCOLS, col_names, col_types);
  if (rc != 0) LF_ERROR(("Error creating linecard table"));

  fdp->linecard_tbl = db_open_table(fdp->database, LF_TABLE_LINECARDS);
  if (fdp->linecard_tbl == NULL) LF_ERROR(("Error opening linecard table"));

  return 0;

 except:
  return -1;
}

/*
 * Flush the DB fabric
 */
int
lf_flush_fabric_db(
  struct lf_fabric_db *fdp)
{
  int rc;

  if (fdp->host_tbl != NULL) {
    rc = db_flush_table(fdp->host_tbl);
    if (rc != 0) LF_ERROR(("Flushing host table"));
  }

  if (fdp->nic_tbl != NULL) {
    rc = db_flush_table(fdp->nic_tbl);
    if (rc != 0) LF_ERROR(("Flushing nic table"));
  }

  if (fdp->link_tbl != NULL) {
    rc = db_flush_table(fdp->link_tbl);
    if (rc != 0) LF_ERROR(("Flushing link table"));
  }

  if (fdp->linecard_tbl != NULL) {
    rc = db_flush_table(fdp->linecard_tbl);
    if (rc != 0) LF_ERROR(("Flushing linecard table"));
  }

  if (fdp->enclosure_tbl != NULL) {
    rc = db_flush_table(fdp->enclosure_tbl);
    if (rc != 0) LF_ERROR(("Flushing enclosure table"));
  }

  return 0;

 except:
  return -1;
}

int
lf_clear_fabric_db(
  struct lf_fabric_db *fdp)
{
  int rc;

  /* clear everything out of any existing database */
  rc = lf_remove_host_from_db(fdp, NULL);
  if (rc != 0) LF_ERROR(("Error clearing host table"));

  rc = lf_remove_nic_from_db(fdp, NULL, -1);
  if (rc != 0) LF_ERROR(("Error clearing NIC table"));

  rc = lf_remove_link_from_db(fdp, NULL, -1, -1, -1);
  if (rc != 0) LF_ERROR(("Error clearing link table"));

  rc = lf_remove_linecard_from_db(fdp, NULL, -1);
  if (rc != 0) LF_ERROR(("Error clearing linecard table"));

  rc = lf_remove_enclosure_from_db(fdp, NULL);
  if (rc != 0) LF_ERROR(("Error clearing enclosure table"));

  return 0;

 except:
  return -1;
}

int
lf_add_fabric_to_db(
  struct lf_fabric_db *fdp,
  struct lf_fabric *fp)
{
  int rc;
  int h;
  int e;

  /* Add hosts and NICs */
  for (h=0; h<fp->num_hosts; ++h) {
    struct lf_host *hp;
    int n;

    hp = fp->hosts[h];

    rc = lf_add_host_to_db(fdp, hp);
    if (rc != 0) LF_ERROR(("Error adding host %s to DB", hp->hostname));

    for (n=0; n<hp->num_nics; ++n) {
      struct lf_nic *nicp;
      int xc;

      nicp = hp->nics[n];

      rc = lf_add_nic_to_db(fdp, nicp);
      if (rc != 0) {
	LF_ERROR(("Error adding NIC %d of host %s to DB",
	      nicp->host_nic_id, hp->hostname));
      }

      /* Now add any NIC-NIC links from this NIC */
      for (xc=0; xc<nicp->num_ports; ++xc) {
        struct lf_xcvr *xcp;
	int p;

	xcp = LF_XCVR(nicp->phys_ports[xc]);

	for (p=0; p<xcp->num_conns; ++p) {
	  struct lf_xcvr *xcp2;

	  xcp2 = LF_XCVR(xcp->ports[p]);

	  /* only print each link once */
	  if (xcp2 != NULL && xcp2->ln_type == LF_NODE_NIC_XCVR) {
	    rc = lf_add_phys_link_to_db(fdp, LF_NODE(xcp), p, nicp->db_status);
	    if (rc != 0) LF_ERROR(("Error adding link from NIC"));
	  }
	}
      }
    }
  }

  /* Add all the enclosures and linecards */
  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    int s;

    ep = fp->enclosures[e];
    rc = lf_add_enclosure_to_db(fdp, ep);
    if (rc != 0) LF_ERROR(("Error adding enclosure %s to DB", ep->name));

    /* add each linecard in the enclosure */
    for (s=0; s<ep->num_lc_slots; ++s) {
      struct lf_linecard *lp;
      int xc;

      lp = ep->slots[s];
      if (lp == NULL) continue;

      rc = lf_add_linecard_to_db(fdp, lp);
      if (rc != 0) {
	LF_ERROR(("Error adding linecard in slot %d of %s to DB",
	      lf_slot_display_no(lp), ep->name));
      }

      /* Now add links from this slot */
      for (xc=0; xc<lp->num_xcvrs; ++xc) {
        struct lf_xcvr *xcp;
	int p;

	xcp = LF_XCVR(lp->xcvrs[xc]);

	for (p=0; p<xcp->num_conns; ++p) {
	  struct lf_xcvr *xcp2;

	  xcp2 = LF_XCVR(xcp->ports[p]);

	  /* only print each link once */
	  if (xcp2 != NULL &&
	      (xcp2->ln_type == LF_NODE_NIC_XCVR ||
	       (xcp2->ln_type == LF_NODE_LC_XCVR && xcp >= xcp2))) {
	    rc = lf_add_phys_link_to_db(fdp, LF_NODE(xcp), p, lp->db_status);
	    if (rc != 0) LF_ERROR(("Error adding link from linecard"));
	  }
	}
      }
    }
  }

  return 0;

 except:
  return -1;
}

/*
 * Re-write the fabric, since some table format has changed
 */
int
lf_rewrite_fabric_db(
  struct lf_fabric_db *fdp,
  struct lf_fabric *fp)
{
  int rc;

  /* Rename the NIC table if open */
  if (fdp->nic_tbl != NULL) {
    rc = db_rename_table(fdp->nic_tbl, LF_TABLE_NICS_PREV);
    if (rc != 0) LF_ERROR(("Cannot rename NIC table"));
    db_close_table(fdp->nic_tbl);
    fdp->nic_tbl = NULL;
  }

  /* Rename the host table if open */
  if (fdp->host_tbl != NULL) {
    rc = db_rename_table(fdp->host_tbl, LF_TABLE_HOSTS_PREV);
    if (rc != 0) LF_ERROR(("Cannot rename host table"));
    db_close_table(fdp->host_tbl);
    fdp->host_tbl = NULL;
  }

  /* Rename the enclosure table if open */
  if (fdp->enclosure_tbl != NULL) {
    rc = db_rename_table(fdp->enclosure_tbl, LF_TABLE_ENCLOSURES_PREV);
    if (rc != 0) LF_ERROR(("Cannot rename enclosure table"));
    db_close_table(fdp->enclosure_tbl);
    fdp->enclosure_tbl = NULL;
  }

  /* Rename the linecard table if open */
  if (fdp->linecard_tbl != NULL) {
    rc = db_rename_table(fdp->linecard_tbl, LF_TABLE_LINECARDS_PREV);
    if (rc != 0) LF_ERROR(("Cannot rename linecard table"));
    db_close_table(fdp->linecard_tbl);
    fdp->linecard_tbl = NULL;
  }

  /* Rename the link table if open */
  if (fdp->link_tbl != NULL) {
    rc = db_rename_table(fdp->link_tbl, LF_TABLE_LINKS_PREV);
    if (rc != 0) LF_ERROR(("Cannot rename link table"));
    db_close_table(fdp->link_tbl);
    fdp->link_tbl = NULL;
  }

  /* re-add everything to the new DB */
  rc = lf_add_fabric_to_db(fdp, fp);
  if (rc != 0) LF_ERROR(("Error recreating fabric DB"));

  /* And flush this to disk */
  rc = lf_flush_fabric_db(fdp);
  if (rc != 0) LF_ERROR(("Error flushing recreated fabric DB"));

  return 0;

 except:
  return -1;
}
